# 📖 How our [framework] works ?

## 📌 Introduction

This framework enables the dynamic loading of scripts and templates in a Chrome extension, managing the modular injection of JavaScript and HTML files, facilitating code reuse and script organization.

## 🖼️ **Icon Management** (`icons.js`)

📜 **Main Function:** `RefreshIcons()`\
🎯 This function refreshes Feather icons in the user interface.

```javascript
export function RefreshIcons() {
  console.log('[commanders_act_assistant:framework:icons:RefreshIcons] Refreshing icons...')
  feather.replace()
  console.log('[commanders_act_assistant:framework:icons:RefreshIcons] All icons refreshed!')
}
```

## 📦 **Framework Entry Point** (`index.js`)

📜 This file exports all framework functionalities and logs a confirmation message upon loading.

```javascript
export * from './registry.js'
export * from './icons.js'
export * from './load.js'

console.log('[commanders_act_assistant:framework:index] All modules loaded.')
```

## 🔄 **Script and Template Loading** (`load.js`)

📜 **Main Features:**

- Dynamically loads scripts referenced in templates
- Injects pages and templates into the DOM
- Manages includes

### 🛠️ `LoadScripts(htmlContent, basePath)`

🔹 Extracts scripts from an HTML content and injects them dynamically.

### 📥 `LoadTemplate(templatePath, callback)`

🔹 Loads an HTML template, executes associated scripts, and applies `onMount` if defined.

### 📃 `LoadPage(name)`

🔹 Loads an entire page into the `#_root` element and manages its includes.

### 📌 `LoadIncludes(htmlContent)`

🔹 Dynamically replaces includes defined in a page with their HTML content and executes their `onMount`.

## 🏗️ **Component Registry Management** (`registry.js`)

📜 **Stores and executes functions associated with loaded templates/pages.**

### 📝 `Register(name, fn)`

🔹 Registers an `onMount` function for a given component.

### 🚀 `Execute(name)`

🔹 Executes the `onMount` function of a registered component if available.

```javascript
export const Registries = {}

export function Register(name, fn) {
  console.log('[commanders_act_assistant:framework:registry:Register]', { name })
  if (typeof fn === 'function') {
    Registries[name] = fn
  } else {
    console.warn(`onMount function for "${name}" is not a valid function`)
  }
}

export function Execute(name) {
  console.log('[commanders_act_assistant:framework:registry:Execute]', { name })
  const onMount = Registries[name]
  if (onMount) {
    onMount()
  } else {
    console.warn(`No onMount function found for "${name}"`)
  }
}
```

## 📂 **Registry Usage Example**

### 📝 `includes/navigation/index.js`

This file registers an `onMount` function that initializes a logout button.

```javascript
import { Register } from '../../framework/index.js'
import { logout } from '../../services/index.js'

function onMount() {
  console.log('includes:navigation:onMount')
  initButtons()
}

function initButtons() {
  console.log('includes:navigation:initButtons')
  const logoutButton = document.getElementById('logoutButton')

  logoutButton.addEventListener('click', async () => {
    console.log('navigation:click:logout')
    await logout()
  })
}

Register('includes/navigation', onMount)
```

### 📌 `includes/navigation/index.html`

```html
<nav class="navbar navbar-expand-lg bg-body-tertiary">
  <div class="container-fluid">
    <a class="navbar-brand" href="https://www.commandersact.com" target="_blank">
      <img src="images/logo_full.svg" alt="Logo" height="48" class="d-inline-block align-text-top">
    </a>
    <div class="d-flex">
      <button id="logoutButton" class="btn btn-primary">
        Logout
      </button>
    </div>
  </div>
</nav>
<!-- script('index') -->
```

📌 **Explanation**:

- `onMount` is automatically executed after the script loads.
- `initButtons` adds an event listener to the logout button.
- The `<!-- script('index') -->` line instructs the framework to load `index.js`.
